const SEARCH_PROVIDER_BASE = 'https://duckduckgo.com/';

function buildSearchUrl(q) {
    return `${SEARCH_PROVIDER_BASE}?q=${encodeURIComponent(q)}`;
}

function hasScheme(s) {
    return /^[a-zA-Z][a-zA-Z\d+\-.]*:\/\//.test(s);
}

function isHttpProtocol(u) {
    return u.protocol === 'http:' || u.protocol === 'https:';
}

function isIPv4(host) {
    const parts = host.split('.');
    if (parts.length !== 4) return false;
    for (const p of parts) {
        if (!/^\d{1,3}$/.test(p)) return false;
        const n = Number(p);
        if (n < 0 || n > 255) return false;
    }
    return true;
}

function isLikelyDomain(host) {
    if (!host.includes('.')) return false;
    if (host.length > 253) return false;
    const labels = host.split('.');
    if (labels.some(l => l.length === 0 || l.length > 63)) return false;
    const labelRE = /^[A-Za-z0-9](?:[A-Za-z0-9-]{0,61}[A-Za-z0-9])$/;
    if (labels.some(l => !labelRE.test(l))) return false;
    const tld = labels[labels.length - 1];
    return /^[A-Za-z]{2,63}$/.test(tld);
}

function isNavigable(input) {
    if (hasScheme(input)) {
        try {
            const u = new URL(input);
            if (!isHttpProtocol(u)) return false;
            if (u.hostname === 'localhost') return true;
            if (isIPv4(u.hostname)) return true;
            if (u.hostname.includes(':') && input.includes('[') && input.includes(']')) return true;
            return isLikelyDomain(u.hostname);
        } catch {
            return false;
        }
    }
    if (input === 'localhost') return true;
    if (isIPv4(input)) return true;
    return isLikelyDomain(input);
}

function normalizeUrl(input) {
    return hasScheme(input) ? input : `https://${input}`;
}

function handleEnter(value) {
    const v = value.trim();
    if (!v) return;
    const target = isNavigable(v) ? normalizeUrl(v) : buildSearchUrl(v);
    window.location.assign(target);
}

function bindSearchBox() {
    const el = document.getElementById('searchBox');
    if (!el) return;
    el.addEventListener('keydown', e => {
        if (e.isComposing || e.key !== 'Enter') return;
        e.preventDefault();
        handleEnter(el.value);
    });
}

if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', bindSearchBox, { once: true });
} else {
    bindSearchBox();
}
